/* -*- Mode:C++; c-file-style:"gnu"; indent-tabs-mode:nil; -*- */
/**
 * NIST-developed software is provided by NIST as a public service. You may
 * use, copy and distribute copies of the software in any medium, provided that
 * you keep intact this entire notice. You may improve, modify and create
 * derivative works of the software or any portion of the software, and you may
 * copy and distribute such modifications or works. Modified works should carry
 * a notice stating that you changed the software and should note the date and
 * nature of any such change. Please explicitly acknowledge the National
 * Institute of Standards and Technology as the source of the software.
 *
 * NIST-developed software is expressly provided "AS IS." NIST MAKES NO
 * WARRANTY OF ANY KIND, EXPRESS, IMPLIED, IN FACT OR ARISING BY OPERATION OF
 * LAW, INCLUDING, WITHOUT LIMITATION, THE IMPLIED WARRANTY OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE, NON-INFRINGEMENT AND DATA ACCURACY. NIST
 * NEITHER REPRESENTS NOR WARRANTS THAT THE OPERATION OF THE SOFTWARE WILL BE
 * UNINTERRUPTED OR ERROR-FREE, OR THAT ANY DEFECTS WILL BE CORRECTED. NIST
 * DOES NOT WARRANT OR MAKE ANY REPRESENTATIONS REGARDING THE USE OF THE
 * SOFTWARE OR THE RESULTS THEREOF, INCLUDING BUT NOT LIMITED TO THE
 * CORRECTNESS, ACCURACY, RELIABILITY, OR USEFULNESS OF THE SOFTWARE.
 *
 * You are solely responsible for determining the appropriateness of using and
 * distributing the software and you assume all risks associated with its use,
 * including but not limited to the risks and costs of program errors,
 * compliance with applicable laws, damage to or loss of data, programs or
 * equipment, and the unavailability or interruption of operation. This
 * software is not intended to be used in any situation where a failure could
 * cause risk of injury or damage to property. The software developed by NIST
 * employees is not subject to copyright protection within the United States.
 */

#ifndef ORAN_CMM_SINGLECOMMANDPERNODE_H
#define ORAN_CMM_SINGLECOMMANDPERNODE_H

#include "oran-cmm.h"

namespace ns3
{

class OranCommand;

/**
 * \ingroup oran
 *
 * A Conflict Mitigation Module that tries to ensure that no two commands
 * issued in a batch affect the same E2 Node. The meaning of "node affected by
 * a command" changes depending on the command. For example, an
 * OranCommandLte2LteHandover has the target E2 Node ID of the eNB where the UE
 * being handovered is attached, but for the purpose of this CMM, the affected
 * node is assumed to be the UE.
 */
class OranCmmSingleCommandPerNode : public OranCmm
{
  public:
    /**
     * Gets the TypeId of the OranCmmSingleCommandPerNode class.
     *
     * \return The TypeId.
     */
    static TypeId GetTypeId();
    /**
     * Creates an instance of the OranCmmSingleCommandPerNode class.
     */
    OranCmmSingleCommandPerNode();
    /**
     * The destructor of the OranCmmSingleCommandPerNode class.
     */
    ~OranCmmSingleCommandPerNode() override;
    /**
     * Prompts this Conflict Mitigation Module to execute its logic
     * and filter the input commands. This module makes sure only one
     * command in each batch affects an E2 node. Note that the E2 node
     * affected may not be the target E2 node (e.g. with a handover command
     * the affected E2 node is the UE being handovered, but the target E2
     * node is the current serving eNB).
     *
     * In case there are multiple commands affecting the same E2 node, the one issued
     * by the default LM takes precedence. If both (or none of the) commands are being
     * issued by the default LM, the first one processed takes precedence.
     *
     * \param inputCommands A map with the input commands generated by all the LMs.
     *
     * \return A vector with the commands that passed the filter.
     */
    std::vector<Ptr<OranCommand>> Filter(
        std::map<std::tuple<std::string, bool>, std::vector<Ptr<OranCommand>>> inputCommands)
        override;

  protected:
    void DoDispose() override;
}; // class OranCmmSingleCommandPerNode

} // namespace ns3

#endif // ORAN_CMM_SINGLECOMMANDPERNODE_H
