/* -*- Mode:C++; c-file-style:"gnu"; indent-tabs-mode:nil; -*- */
/**
 * NIST-developed software is provided by NIST as a public service. You may
 * use, copy and distribute copies of the software in any medium, provided that
 * you keep intact this entire notice. You may improve, modify and create
 * derivative works of the software or any portion of the software, and you may
 * copy and distribute such modifications or works. Modified works should carry
 * a notice stating that you changed the software and should note the date and
 * nature of any such change. Please explicitly acknowledge the National
 * Institute of Standards and Technology as the source of the software.
 *
 * NIST-developed software is expressly provided "AS IS." NIST MAKES NO
 * WARRANTY OF ANY KIND, EXPRESS, IMPLIED, IN FACT OR ARISING BY OPERATION OF
 * LAW, INCLUDING, WITHOUT LIMITATION, THE IMPLIED WARRANTY OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE, NON-INFRINGEMENT AND DATA ACCURACY. NIST
 * NEITHER REPRESENTS NOR WARRANTS THAT THE OPERATION OF THE SOFTWARE WILL BE
 * UNINTERRUPTED OR ERROR-FREE, OR THAT ANY DEFECTS WILL BE CORRECTED. NIST
 * DOES NOT WARRANT OR MAKE ANY REPRESENTATIONS REGARDING THE USE OF THE
 * SOFTWARE OR THE RESULTS THEREOF, INCLUDING BUT NOT LIMITED TO THE
 * CORRECTNESS, ACCURACY, RELIABILITY, OR USEFULNESS OF THE SOFTWARE.
 *
 * You are solely responsible for determining the appropriateness of using and
 * distributing the software and you assume all risks associated with its use,
 * including but not limited to the risks and costs of program errors,
 * compliance with applicable laws, damage to or loss of data, programs or
 * equipment, and the unavailability or interruption of operation. This
 * software is not intended to be used in any situation where a failure could
 * cause risk of injury or damage to property. The software developed by NIST
 * employees is not subject to copyright protection within the United States.
 */

#ifndef ORAN_CMM_H
#define ORAN_CMM_H

#include <ns3/object.h>

#include <map>
#include <string>
#include <vector>

namespace ns3
{

class OranCommand;
class OranNearRtRic;

/**
 * \ingroup oran
 *
 * Base class for the Conflict Mitigation Modules.
 * This base class defines the API for all the Conflict Mitigation
 * Module, and provides implementations for methods that are common,
 * like Activate, Deactivate, IsActive, SetName, and LogLogicToStorage.
 *
 * When the CMM is deactivated, no filtering is applied and all the received
 * Commands are returned for transmission to the nodes.
 *
 * This class cannot be instantiated, as it is missing the implementation
 * of the Filter method.
 */
class OranCmm : public Object
{
  public:
    /**
     * Gets the TypeId of the OranCmm class.
     *
     * \returns The TypeId.
     */
    static TypeId GetTypeId();
    /**
     * Creates an instance of the OranCmm class.
     */
    OranCmm();
    /**
     * The destructor of the OranCmm class.
     */
    ~OranCmm() override;
    /**
     * Activates the Conflict Mitigation Module.
     */
    virtual void Activate();
    /**
     * Deactivates the Conflict Mitigation Module.
     */
    virtual void Deactivate();
    /**
     * Get the active status of this Conflict Mitigation Module.
     *
     * \return True if this Conflict Mitigation Module is active.
     */
    bool IsActive() const;
    /**
     * Get the name of this module.
     *
     * \return The name of this module.
     */
    std::string GetName() const;
    /**
     * Set the name of this module.
     *
     * \param name The name to give this module.
     */
    void SetName(const std::string &name);
    /**
     * Prompts this Conflict Mitigation Module to execute its logic
     * and filter the input commands. This will only work if the
     * Module is active; otherwise, the input set of commands will be
     * returned without filtering.
     *
     * \param inputCommands A map with the input commands generated by all the LMs.
     *
     * \return A vector with the commands that passed the filter.
     */
    virtual std::vector<Ptr<OranCommand>> Filter(
        std::map<std::tuple<std::string, bool>, std::vector<Ptr<OranCommand>>> inputCommands) = 0;

  protected:
    /**
     * Disposes of the object.
     */
    void DoDispose() override;
    /**
     * Log a message to the data storage.
     *
     * \param msg The message to log.
     */
    void LogLogicToStorage(const std::string &msg) const;

    /**
     * Pointer to the Near RT-RIC.
     */
    Ptr<OranNearRtRic> m_nearRtRic;
    /**
     * Flag to keep track of active status.
     */
    bool m_active;
    /**
     * Flag to indicate if logic should be logged to data storage.
     */
    bool m_verbose;
    /**
     * The name of this module.
     */
    std::string m_name;
}; // class OranCmm

} // namespace ns3

#endif // ORAN_CMM_H
